/*********************************************************************
**  Device: A9139M0
**  File:		main.c
**  Target:		
**  Tools:		
**  Updated:	2025-06-27
**  Description: FIFO
**  This file is a sample code for your reference.
**
**  Copyright (C) 2025 AMICCOM Corp.
**
*********************************************************************/
#include <stdio.h>
#include "system_A9139M0.h"
#include "Delay.h"
#include "A9139M0_RFLIB.h"

/********************************************************************
** Configure Select
*********************************************************************/
#define A40_DR9K6_IFBW100K    0
#define A40_DR50K_IFBW50K     1
#define A40_DR100K_IFBW100K   2
#define A40_DR150K_IFBW150K   3
#define A40_DR250K_IFBW250K   4
#define A40_DR500K_IFBW250K   5
#define A40_DR2M_IFBW250K     6

/********************************************************************/
/*            User Select Data Rate & Crystal
=====================================================================
*             note: Data Rate 9.6Kbps Xtal = 12.8MHz
*                   Data Rate 50Kbps  Xtal = 12.8MHz
*                   Data Rate 100Kbps Xtal = 12.8MHz
*                   Data Rate 150Kbps Xtal = 19.2MHz
*                   Data Rate 250Kbps Xtal = 16MHz
*                   Data Rate 500Kbps Xtal = 16MHz 
*                   Data Rate 2Mbps   Xtal = 16MHz    
*/

#define SelectConfig          A40_DR250K_IFBW250K  // User define

/*********************************************************************/


/*********************************************************************
**  I/O Declaration
*********************************************************************/
#define GIO1    GPIO0->DATA_PIN.P13 		//GIO1
#define GIO2 		GPIO0->DATA_PIN.P14 		//GIO2
#define CKO     GPIO0->DATA_PIN.P15 		//CKO

//CKO
#define DCK_CKO         0
#define RCK_CKO         1
#define FPF_CKO         2
#define PF8M_CKO        9
#define ROSC_CKO        10
#define EOADC_CKO       11
#define OKADCN_CKO      12
#define VPOAK_CKO      	15

//GIO1
#define ARCWTR_GPIO1    0
#define EOAC_GPIO1      1
#define FSYNC_GPIO1 		1
#define TMEO_GPIO1      2
#define CD_GPIO1        2
#define PMDO_GPIO1      3
#define TWWS_GPIO1      4
#define CWTR_GPIO1      4
#define SDO_GPIO1      	6
#define TRXD_GPIO1      7
#define RXD_GPIO1      	8
#define TXD_GPIO1      	9
#define PDNRX_GPIO1     10
#define FPF_GPIO1     	13
#define PDNTX_GPIO1     14
#define FMTDO_GPIO1     15

//GIO2
#define ARCWTR_GPIO2    0
#define EOAC_GPIO2      1
#define FSYNC_GPIO2     1
#define TMEO_GPIO2      2
#define CD_GPIO2        2
#define PMDO_GPIO2      3
#define TWWS_GPIO2      4
#define CWTR_GPIO2      4
#define PDNPA_GPIO2     6
#define TRXD_GPIO2      7
#define RXD_GPIO2      	8
#define TXD_GPIO2      	9
#define VPOAK_GPIO2     12
#define FPF_GPIO2     	13
#define BDF_GPIO2     	14

/*********************************************************************
**  Constant Declaration
*********************************************************************/
#define TIMEOUT     				100	//100ms	

//ID
#define ID_2BYTES   				0x00
#define ID_4BYTES    				0x01
#define ID_6BYTES    				0x02
#define ID_8BYTES    				0x03

//INTTERRUPT
#define ENABLE_RFINT_WTR    0x01
#define ENABLE_RFINT_WAKEBB 0x02
#define ENABLE_RFINT_FPF    0x04
#define ENABLE_RFINT_FSYNC  0x08
#define ENABLE_RFINT_VPOAK  0x10
#define RADIO_INT_CLEAR		  0x40
#define RFINT_WTR_STATE	    0x01
#define RFINT_WAKEBB_STATE  0x02
#define RFINT_FPF_STATE     0x04
#define RFINT_FSTNC_STATE   0x08
#define RFINT_VPOAK_STATE   0x10

//WOR & WOT
#define BY_FRAME_SYNC_OK    	0x00
#define BY_PREAMBLE_DETECT_OK	BIT28
#define BY_CARRIER_DETECT_OK 	BIT29
#define WOR									0
#define WOT 								1
#define DUTY_BY_AC					0
#define DUTY_BY_SL					1

//SLEEPER TIMER
#define CLR_TMRIF		  			BIT5

//ADC & RSSI
#define ADCI_CLEAR		  		BIT16

#define AVSEL_NO_AVERAGE			( 0x00 << 10 )
#define AVSEL_AVERAGE_2				( 0x01 << 10 )
#define AVSEL_AVERAGE_4				( 0x02 << 10 )
#define AVSEL_AVERAGE_8				( 0x03 << 10 )

#define MVSEL_AVERAGE_8				( 0x00 << 8 )
#define MVSEL_AVERAGE_16			( 0x01 << 8 )
#define MVSEL_AVERAGE_32			( 0x02 << 8 )
#define MVSEL_AVERAGE_64			( 0x03 << 8 )

#define ADC_SOURCE_CLK_DIV_4	( 0x00 << 5 )
#define ADC_SOURCE_CLK_DIV_8	( 0x01 << 5 )
#define ADC_SOURCE_CLK_DIV_16	( 0x02 << 5 )
#define ADC_SOURCE_CLK_DIV_32	( 0x03 << 5 )

#define ADC_NO_AVERAGE			   ( 0x00 << 1 )
#define ADC_2_TIMES_AVERAGE	   ( 0x01 << 1 )
#define ADC_4_TIMES_AVERAGE	   ( 0x02 << 1 )
#define ADC_8_TIMES_AVERAGE	   ( 0x03 << 1 )
#define ADC_16_TIMES_AVERAGE	 ( 0x04 << 1 )
#define ADC_32_TIMES_AVERAGE	 ( 0x05 << 1 )
#define ADC_64_TIMES_AVERAGE	 ( 0x06 << 1 )
#define ADC_128_TIMES_AVERAGE	 ( 0x07 << 1 )

//AES_CCM
#define CCM_TX_ON						0x01
#define AES_GO 							0x01
#define CCM_RX_ON						0x20
#define RXDE_GO							0x40
#define TXEN_GO							0x80		
#define CCMI_STATE					0x10
#define AESI_STATE					0x20
#define AESCCM_Int_Clear		0x10
#define CCM_FAIL						0x10
#define CCM_OK							0x20

//CACHE
#define CACHE_DISABLED			0x00
#define CACHE_ENABLING 			0x01
#define CACHE_ENABLED				0x02
#define CACHE_DIABLING			0x03

/*********************************************************************
**  Global Variable Declaration
*********************************************************************/
Uint8	 volatile   RF_Flag;     
Uint8	 volatile		SLPTIMER0_Flag;
Uint8	 volatile		SLPTIMER1_Flag;
Uint8	 volatile	  Flag_Master;
Uint8	 volatile   TmpBuf[64];
Uint8  volatile   CalResult;
Uint8  volatile   timer;
Uint8	 volatile   TimeoutFlag;
Uint8  volatile  	CmdBuf[13];
Uint8  volatile  	UartSendCnt;
Uint8  volatile  	*Uartptr;
Uint8  volatile   ConfigType;
Uint16 volatile   RxCnt;
Uint16 volatile	  LossCnt;
Uint16 volatile	  ErrCnt;
Uint16 volatile  	TimerCnt;
Uint32 volatile	  Err_ByteCnt;
Uint32 volatile	  Err_BitCnt;
Uint8  volatile		ADC12B_Flag;
Uint32 volatile   CL_tmp;
Uint8	 volatile		RF_AES_Flag;
Uint8	 volatile	  RF_CCM_Flag	;

const Uint8 ID_TAB[8] = { 0x34, 0x75, 0xC5, 0x2A, 0xC7, 0x33, 0x45, 0xEA }; //ID code
const	Uint8	BITCOUNT_TAB[16] = { 0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4 };
const	Uint8	PN9_TAB[] =
{   
	  0xFF, 0x83, 0xDF, 0x17, 0x32, 0x09, 0x4E, 0xD1,
    0xE7, 0xCD, 0x8A, 0x91, 0xC6, 0xD5, 0xC4, 0xC4,
    0x40, 0x21, 0x18, 0x4E, 0x55, 0x86, 0xF4, 0xDC,
    0x8A, 0x15, 0xA7, 0xEC, 0x92, 0xDF, 0x93, 0x53,
    0x30, 0x18, 0xCA, 0x34, 0xBF, 0xA2, 0xC7, 0x59,
    0x67, 0x8F, 0xBA, 0x0D, 0x6D, 0xD8, 0x2D, 0x7D,
    0x54, 0x0A, 0x57, 0x97, 0x70, 0x39, 0xD2, 0x7A,
    0xEA, 0x24, 0x33, 0x85, 0xED, 0x9A, 0x1D, 0xE0
};	// This table are 64bytes PN9 pseudo random code.

//User defined AES_CCM
const Uint8 CCM_FIFO[23] = { 0x10, 0x90, 0x50, 0xD0, 0x30, 0xB0, 0x70, 0xF0,
                             0x08, 0x88, 0x48, 0xC8, 0x28, 0xA8, 0x68, 0xE8,
                             0x18, 0x98, 0x58, 0xD8, 0x38, 0xB8, 0x78 };

const Uint32 AES_KEY_IN[4] = { 0xC0C1C2C3, 0xC4C5C6C7, 0xC8C9CACB, 0xCCCDCECF };

const Uint32 NONCE[4][5]={
	{ 0x00000000, 0x01020304, 0x05060700, 0x00000000, 0x00000804 },	//level 4
	{ 0x00000000, 0x01020304, 0x05060700, 0x00000000, 0x00000805 },	//level 5
	{ 0x00000000, 0x01020304, 0x05060700, 0x00000000, 0x00000806 },	//level 6
	{ 0x00000000, 0x01020304, 0x05060700, 0x00000000, 0x00000807 }	//level 7
};

//Register for user
Uint32	volatile	CLOCK2_Reg;
Uint32	volatile 	CODE1_Reg;
Uint32	volatile	FIFO_Reg;
Uint32	volatile	TXPWR_Reg;
Uint32	volatile	MODEC2_Reg;
Uint32	volatile	TXCTL_Reg; 
Uint32	volatile	GPIO_Reg;
Uint32	volatile	PLL1_Reg; 
Uint32	volatile	PLL7_Reg;
Uint32	volatile	PWRCTL2_Reg;
Uint32 	volatile	FPCTL_Reg;
Uint32 	volatile	WORT2_Reg;
Uint32 	volatile	DSSS1_Reg;
Uint32 	volatile	ADC_Reg;
Uint32 	volatile	RTH_Reg;
Uint32 	volatile	ADC12B_CTL_Reg;
Uint32 	volatile	RCCTRL1_Reg;
Uint32 	volatile	RCCTRL2_Reg;
Uint32 	volatile	INTIE_Reg;
Uint32 	volatile	SYN_Reg;
Uint32 	volatile	SYCK_Reg;
Uint32 	volatile	RXCTL_Reg;

//Backup IO 
Uint32	volatile  OUTEN_Reg;
Uint32	volatile  PUN_Reg;
Uint32	volatile  WUN_Reg;
Uint32	volatile  ALTFUNSET_Reg;

Uint8 TXbuf[151]={0};
Uint8 RXbuf[151]={0};
Uint8 volatile  UartRecevicCnt;

/*********************************************************************
**  function Declaration
*********************************************************************/
void InitMCU ( void );
void InitTimer0 ( void );
void InitUart0 ( void );
void WriteFIFO ( const Uint8* buf, Uint8 size );
void RxPacket ( void );
Uint8 WriteID ( void );
void Err_State( void );
void RCOSC_Cal ( void );
void RTC_Enable ( void );
void WOR_enable_by_preamble ( void );
void WOR_enable_by_sync ( void );
void WOR_enable_by_carrier ( void );
void WOT_enable ( void );
void TWWS_enable ( void );
void RSSI_measurement ( void );
void RF_SetBBIO ( Uint32 gio1s, Uint32 gio2s, Uint32 ckos, Uint8 sw );
void UserRegister ( void );
void Reg_Init ( void );
void IDLength ( Uint32 idl );
void TXPower ( Uint32 tbg, Uint32 tdc, Uint32 pac );
void FIFOLength ( Uint32 fep );
void TXModeSel ( Uint8 mode );
void TXModulation ( Uint8 tme );
void CrystalSelect ( void );
void WDTReset ( void );
void AES_CCM_mode ( void );
void ACK_mode ( void );
void ACK_FIFO_mode ( void );
void FCC ( void );
void EnableCache ( void );
void DisableCache ( void );
void EnableHS ( void );
void DisableHS ( void );

/*********************************************************************
* main loop
*********************************************************************/
int main ( void )
{	
	Uint8 i=0;
		/*
		RX sensitivity may degrade 3~5dBm when supply voltage REGI is lower than 2.7V.
		Please consult with AMICCOM FAE for detail.
		*/

		while ( ( GPIO0->DATA & BIT12 ) == 0 ); //Wait Debug Pin(P0.12) Go High
		ConfigType = SelectConfig;

		//Initial HW
    InitMCU();
		CrystalSelect();
	  InitTimer0(); //Timer0 Initial
    InitUart0();//UART0 Initial
		NVIC_EnableIRQ ( UART0_IRQn );	    
	  NVIC_EnableIRQ ( TIMER0_IRQn );	
		
		if ( GPIO0->DATA_PIN.P08 == 1 )
		    Flag_Master = 1;
	  else
		    Flag_Master = 0;	
	
		RFLIB_Reset ( RF_RST );
		
		// If want to set other c-load, please check with our FAE engineer.
//		RFLIB_InitCrystalCL ( 1, 28 ); //INTCXC = 1, XCL = 28, for 49U DIP C-laod 20pF
		RFLIB_InitCrystalCL ( 0, 23 ); //INTCXC = 0, XCL = 23, for SMD3225 C-load 12pF	
		
		Delay100us ( 5 ); //Delay 500us For Regulator Stabilized 
		
		CalResult = RFLIB_InitRF();
    if ( CalResult ) //Check Return Value In A9139M0_RFLIB.h 
		    Err_State(); 
		
		RFLIB_StrobeCmd ( CMD_STBY );
		
		//User define
		UserRegister();		

		/*                    
    How to set the MCU clock to High Speed Mode 
    Step1 : HS Enable(XREF_PLL = 51.2MHz)
    Step2 : Cache Enable(Improve in code fetching performance)

    Note : For Data Rate = 100Kbps / Xtal = 12.8MHz / XREF_PLL = 51.2MHz 
    If user wanna change CFG, Xtal or XREF_PLL Frequency, please consult with AMICCOM FAE for detail.
    */                   
    //EnableHS();                     
		//EnableCache();
                     
    /* How to switch MCU clock from High Speed Mode to normal mode.
		Step1 : Cache Disable
		Step2 : HS Disable(XREF_PLL = 12.8MHz)
    */
		//DisableCache();
    //DisableHS();
		
		RFLIB_SetCH ( 0 );//430.001MHz
    /* 
    How To Set RF Channel
    RF Base Freq : 430.001MHz
    Channel Step : 0.1MHz
    SetCH        : 10
    Range        : 430.001MHz ~ 434.701MHz
    RF Frequency = RF base Freq + (Channel Step * SetCH) = 430.001MHz + ( 0.1MHz * 10 ) = 431.001MHz 
    */		
		
		//AES_CCM_mode();
		//ACK_mode();
		//ACK_FIFO_mode();
		
	  if ( Flag_Master ) //P0.8 = 1
	  {
		    NVIC_EnableIRQ( RADIO_IRQn );
				INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_WTR; 
				RADIO->RFINT_WRITE = INTIE_Reg; 
			
			
//				TXPower ( Uint32 tbg, Uint32 tdc, Uint32 pac );
				TXPower(63, 1, 2);  //TX power 19.9dBm
		  
				if ( GPIO0->DATA_PIN.P00 == 0 )
				{
					TXPower(42, 1, 2);  //TX power 9.9dBm
				}
				
				if( GPIO0->DATA_PIN.P01 == 0 )
				{
					TXPower(28, 1, 2);  //TX power 5.0dBm
				}
//				for(i=0;i<151;i++)
//				{
//					TXbuf[i]=i;
//				}
				UartRecevicCnt=0;
				
				while ( 1 )
				{					
						while(UartRecevicCnt!=151);
						RF_SetBBIO ( ARCWTR_GPIO1, EOAC_GPIO2, DCK_CKO, ENABLE );
//						WriteFIFO ( PN9_TAB, sizeof ( PN9_TAB ) ); //Write data to TX FIFO
						WriteFIFO ( TXbuf, sizeof ( TXbuf ) ); //Write data to TX FIFO
						UartRecevicCnt=0;
						RF_Flag = 0;
						RFLIB_StrobeCmd ( CMD_TX ); //Entry TX & Transmit						
						while ( RF_Flag == 0 ); //Wait TX Finish
						Delay100us ( 2 ); //delay 200us for TX ramp down
						
//					  RF_SetBBIO ( ARCWTR_GPIO1, FSYNC_GPIO2, RCK_CKO, ENABLE );
//						timer = 0;
//						TimeoutFlag = 0;
//						RF_Flag = 0;
//						RFLIB_StrobeCmd ( CMD_RX ); //Into RX Mode
//						Delay10us ( 1 );
//            while ( ( RF_Flag == 0 ) && ( TimeoutFlag == 0 ) ); //Wait RX Finish Or Timer0 Timeout
//						if ( TimeoutFlag )
//						{
//								RFLIB_StrobeCmd ( CMD_STBY ); //Exit RX Mode
//								LossCnt++;
//						}
//						else
//            {								
//						    RxPacket(); //Receive Data And Compare With PN9
//						    Delay1ms ( 10 );
//						}	
		    }				
    }
	  else
	  {		
				NVIC_EnableIRQ ( RADIO_IRQn );
				INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_WTR; 
				RADIO->RFINT_WRITE = INTIE_Reg;
			
				RxCnt = 0;
				ErrCnt = 0;
			  Err_BitCnt = 0;
				LossCnt = 0;

				while ( 1 )
				{
						RF_SetBBIO( ARCWTR_GPIO1, FSYNC_GPIO2, RCK_CKO, ENABLE );
						timer = 0;
						TimeoutFlag = 0;
						RF_Flag = 0;
						RFLIB_StrobeCmd ( CMD_RX ); //Into RX Mode
						Delay10us ( 1 );
						while (  RF_Flag == 0 );
						RxPacket(); //Receive Data And Compare With PN9
					
						GPIO0->DATAOUT_PIN.P09=~GPIO0->DATAOUT_PIN.P09;
					
						UartSendCnt = 151;
						Uartptr = &RXbuf[0];
						UART0->DATA = RXbuf[0];
						while(UartSendCnt);
					
//						while ( ( RF_Flag == 0 ) && ( TimeoutFlag == 0 ) ); //Wait RX Finish Or Timer0 Timeout
//						if ( TimeoutFlag )
//						{
//								RFLIB_StrobeCmd ( CMD_STBY ); //Exit RX Mode
//								LossCnt++;
//						}
//						else
//						{								
//						    RxPacket(); //Receive Data And Compare With PN9
//						    Delay1ms ( 10 );
//						}						
//					
//						RF_SetBBIO( ARCWTR_GPIO1, EOAC_GPIO2, DCK_CKO, ENABLE ); 
//						WriteFIFO( PN9_TAB, sizeof( PN9_TAB ) ); //write data to tx fifo
//						RF_Flag = 0;
//						RFLIB_StrobeCmd( CMD_TX ); //entry tx     						
//						while( RF_Flag == 0 ); //wait transmit completed						
//						Delay100us ( 2 ); //delay 200us for TX ramp down
        }
    }
}

/*********************************************************************
** InitMCU
*********************************************************************/
void InitMCU( void )
{		
		GPIO0->OUTENABLESET = 0xFFFFFEFC;	//P0.8 	
		GPIO0->PUN = 0xFFFFFFFF;
		GPIO0->WUN = 0xFFFFFFFF;		
}

/************************************************************************
**  InitTimer0
************************************************************************/
void InitTimer0( void )
{
    NVIC_DisableIRQ( TIMER0_IRQn );
	  TIMER0->RELOAD = SystemCoreClock / 1000;
	  TIMER0->VALUE = SystemCoreClock / 1000; //1ms  
	  TIMER0->INTCLEAR = 0x01; //clear timer interrupt
		TIMER0->CTRL = BIT0 | BIT3; //timer enable & interrupt
}

/************************************************************************
**  Init Uart0
************************************************************************/
void InitUart0( void )
{
    Uint32 baudrate;
		baudrate = 115200; 
		UART0->BAUDDIV = SystemCoreClock / baudrate; 
	  UART0->CTRL = 0x0F; 
	  GPIO0->ALTFUNCSET |=  BIT16 | BIT17;
	  NVIC_EnableIRQ( UART0_IRQn );
}

/*********************************************************************
** WriteFIFO
*********************************************************************/
void WriteFIFO( const Uint8* buf, Uint8 size )
{
    Uint32 i;
		RFLIB_Reset( TXPOINT_RST );	
		for ( i = 0; i < size; i++ )
		    RADIO->TX_DATA[i] = buf[i];
}

/*********************************************************************
** RxPacket
*********************************************************************/
//void RxPacket( void )
//{
//    Uint8_ i,recv,tmp;
//	  Uint8 err;
//	  err=0;
//	
//    RxCnt++;
//	
//		RFLIB_Reset( RXPOINT_RST );	
//	
//    for( i = 0; i < 64; i++ )
//    {
//        recv = RADIO->RX_DATA[i];
//        TmpBuf[i] = recv;		
//        if( ( recv ^ PN9_TAB[i] ) != 0 )
//        {
//            Err_ByteCnt++;
//            tmp = recv ^ PN9_TAB[i];
//            Err_BitCnt += ( BITCOUNT_TAB[tmp>>4] + BITCOUNT_TAB[tmp&0x0F] );  
//						err=1;					
//        }
//    }
//		
//	  if( err ) //packet error
//        ErrCnt++;
//}


void RxPacket( void )
{
    Uint8_ i,recv,tmp;
//	  Uint8 err;
//	  err=0;
	
    RxCnt++;
	
	RFLIB_Reset( RXPOINT_RST );	
	
    for( i = 0; i < 151; i++ )
    {
        RXbuf[i] = RADIO->RX_DATA[i];
    }
		

}
/************************************************************************
**  WriteID
************************************************************************/
Uint8 WriteID ( void )
{
    Uint32 tmp1,tmp2;

    //Write ID
    tmp1 = ID_TAB[0]<<24 | ID_TAB[1]<<16 | ID_TAB[2]<<8 | ID_TAB[3];
    RFLIB_WriteID(tmp1);   
   
    //Read ID   
    tmp2 = RFLIB_ReadID();  
   
    //check ID
    if ( tmp1 != tmp2 )
        return FAIL;

    return PASS;
}

/************************************************************************
**  Err_State
************************************************************************/
void Err_State ( void )
{
    //ERR display
    //Error Proc...
    //...
	
		/* Watchdog reset */
		//WDTReset();	
                     

    while ( 1 );
}

/************************************************************************
**  RCOSC_Cal
************************************************************************/
void RCOSC_Cal  ( void )
{
    Uint16 tgnum, tmp;
	
    switch (ConfigType) 
    {	
				case 3: //19.2   
				tgnum = 1172;
				break;
							
				case 4://16		
				case 5:
				case 6:
				tgnum = 977;
				break;
							
				case 0://12.8
				case 1:
				case 2:
				tgnum = 781;
				break;    
    }			
		tgnum &= 0xFFF;
		RCCTRL2_Reg = ( ( RCCTRL2_Reg & ~0x001F0000 ) | 0x0001B000 ); //b16~18:RCOT=3, b19~b20:MVS=1/16
	  POWER->RCCTRL2 = RCCTRL2_Reg;
	  POWER->RCTGM = tgnum;
	
		while ( 1 )
		{
		    do {	
						RCCTRL1_Reg = ( ( RCCTRL1_Reg & ~0x10000003 ) | BIT0 | BIT1 | BIT28 ); //b1:TMRE=1, b2:RCTS=1, b28:SWM=1
				    POWER->RCCTRL1 = RCCTRL1_Reg;						
				} while ( ( POWER->RCCTRL1 & BIT0 ) == 0x00 ); //b0:ENCAL, Encal Auto Clear to 0
				
		    tmp = POWER->RCTGM;
		
		    if ( tmp > ( tgnum - 10 ) && tmp < ( tgnum + 10 ))
			      break;
	  }		
}

/************************************************************************
**  RTC_Enable
************************************************************************/
void RTC_Enable ( void )
{   
		Uint8 rtcap;
		
		rtcap = 16;		
		GPIO0->PUN &= ( ~( BIT22 | BIT23 ) );
		GPIO0->OUTENABLECLR = BIT22 | BIT23;
		POWER->PWR_TA = 0xAAAAAAAA;
    POWER->PWR_TA = 0x55555555;				
		POWER->IO_CTRL2 = 0x0000003F; 		
		RCCTRL1_Reg = ( ( RCCTRL1_Reg & ~0x00F88016 ) | BIT1 | BIT2 | BIT4 | BIT15 | ( ( rtcap & 0x1F ) << 19 ) ); //b1:TMRE=1, b2:RCTS=1, b4:MAN=1, b15:RLVS=1, b19~b23:RTCAP=16
		POWER->RCCTRL1 = RCCTRL1_Reg; 
		RCCTRL2_Reg = ( ( RCCTRL2_Reg & ~0x000703FF ) | 0x00040012 ); //b0~9:MRCT=18, b16~18:RCOT=4
		POWER->RCCTRL2 = RCCTRL2_Reg;
		Delay1ms(300); //RC settling time 
}

/************************************************************************
**  WOR_enable_by_preamble
************************************************************************/
void WOR_enable_by_preamble ( void )
{    
		Uint16 wrdly, wordly;
	
		wrdly = 256;
		wordly = 127;
	
		RCOSC_Cal(); //internal RC OSC CAL
    //RTC_Enable();   //use external RC OSC

		RF_SetBBIO ( PMDO_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE ); 
		RF_SetBBIO ( TWWS_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE ); 

		//setup WOR Sleep time and Rx time
		//WOR Active Period = (WRDLY[8:0]+1) x (1/32768), (30.5us ~ 15.6ms).
		//WOR Sleep  Period = (WORDLY[9:0]+1) x (32/4096), (7.8ms ~ 7.99s).
 	 	//WRDLY[8:0]=256 : WOR Active Period ~ 7.84ms, WORDLY[9:0]=127 : WOR Sleep  Period ~ 1s
		WORT2_Reg = ( ( WORT2_Reg & ~0x03FF01FF ) | ( wrdly & 0x1FF ) | ( ( wordly & 0x3FF ) << 16 ) );	
	
		//WOR_HOLD : WOR hold RX setting when preamble/sync/carrier detected ok.
		//      [0]: No hold. [1]: Hold RX.
		//WOR_SEL: TWWS=1 setting.		
		//      [0]:RX valid packet.  [1]: preamble/sync/carrier  detected ok.
		//WOR_RST: Shall be set to [1].				
		WORT2_Reg = ( ( WORT2_Reg & ~0x03000000 ) | BY_PREAMBLE_DETECT_OK );
		WORT2_Reg = ( ( WORT2_Reg & ~0x04000000 ) | BIT30 ); //b30:WOR_SEL=1	
		RADIO->WORT_Item.TSEL = ENABLE; 
		DSSS1_Reg = ( ( DSSS1_Reg & ~0x03000000 ) | BIT24 | BIT25 );	//b24:WOR_HOLD=1, b25:WOR_RST=1
		RADIO->WORT2 = WORT2_Reg;
		RADIO->DSSS1 = DSSS1_Reg;
		
		RFLIB_StrobeCmd ( CMD_STBY );

		NVIC_EnableIRQ ( RADIO_IRQn );//enable RF interrupt
		INTIE_Reg	= RADIO_INT_CLEAR | ENABLE_RFINT_WAKEBB; 
		RADIO->RFINT_WRITE = INTIE_Reg;		
		
		
		RADIO->WORT_Item.WN = 0;
		RADIO->WORT_Item.WOR_EN = ENABLE;//enable WOR function		
            
		//WOR_SEL=1 : preamble/sync/carrier detected ok -> TWWS go high -> wake up MCU
		//defined process by user
		//...
		//...
		//RFLIB_StrobeCmd ( CMD_STBY );  //TWWS reset by strobe command.
}

/************************************************************************
**  WOR_enable_by_sync
************************************************************************/
void WOR_enable_by_sync ( void )
{
		Uint16 wrdly, wordly;
	
		wrdly = 256;
		wordly = 127;
		RCOSC_Cal(); //internal RC OSC CAL
    //RTC_Enable();   //use external RC OSC

		RF_SetBBIO ( FSYNC_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE ); 
		RF_SetBBIO ( TWWS_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE ); 

		//setup WOR Sleep time and Rx time
		//WOR Active Period = (WRDLY[8:0]+1) x (1/32768), (30.5us ~ 15.6ms).
		//WOR Sleep  Period = (WORDLY[9:0]+1) x (32/4096), (7.8ms ~ 7.99s).
 	 	//WRDLY[8:0]=256 : WOR Active Period ~ 7.84ms, WORDLY[9:0]=127 : WOR Sleep  Period ~ 1s
		WORT2_Reg = ( ( WORT2_Reg & ~0x03FF01FF ) | ( wrdly & 0x1FF ) | ( ( wordly & 0x3FF ) << 16 ) );	

		//WOR_HOLD : WOR hold RX setting when preamble/sync/carrier detected ok.
		//      [0]: No hold. [1]: Hold RX.
		//WOR_SEL: TWWS=1 setting.
		//      [0]:RX valid packet.  [1]: preamble/sync/carrier  detected ok.
		//WOR_RST: Shall be set to [1].
		WORT2_Reg = ( ( WORT2_Reg & ~0x03000000 ) | BY_FRAME_SYNC_OK );
		WORT2_Reg = ( WORT2_Reg & ~0x04000000 ); //b30:WOR_SEL=0	
		RADIO->WORT_Item.TSEL = ENABLE; 
		DSSS1_Reg = ( ( DSSS1_Reg & ~0x03000000 ) | BIT24 | BIT25 );	//b24:WOR_HOLD=1, b25:WOR_RST=1
		RADIO->WORT2 = WORT2_Reg;
		RADIO->DSSS1 = DSSS1_Reg;
		
		RFLIB_StrobeCmd ( CMD_STBY );

		NVIC_EnableIRQ ( RADIO_IRQn );//enable RF interrupt
		INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_WAKEBB; 
		RADIO->RFINT_WRITE = INTIE_Reg;	
		

		RADIO->WORT_Item.WN = 0;
		RADIO->WORT_Item.WOR_EN = ENABLE;//enable WOR function
               
		//WOR_SEL=0 : RX valid packet -> TWWS go high -> wake up MCU
		//check data
		RxPacket();
		RFLIB_StrobeCmd ( CMD_STBY ); //TWWS reset by strobe command.
}

/************************************************************************
**  WOR_enable_by_carrier
************************************************************************/
void WOR_enable_by_carrier ( void )
{
		Uint8 rth;
		Uint16 wrdly, wordly;
	
		rth = 110;
		wrdly = 256;
		wordly = 127;
	
		RCOSC_Cal(); //internal RC OSC CAL
    //RTC_Enable();   //use external RC OSC

		RF_SetBBIO ( CD_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE );
		RF_SetBBIO ( TWWS_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE ); 

		//setup WOR Sleep time and Rx time
		//WOR Active Period = (WRDLY[8:0]+1) x (1/32768), (30.5us ~ 15.6ms).
		//WOR Sleep  Period = (WORDLY[9:0]+1) x (32/4096), (7.8ms ~ 7.99s).
 	 	//WRDLY[8:0]=256 : WOR Active Period ~ 7.84ms, WORDLY[9:0]=127 : WOR Sleep  Period ~ 1s
		WORT2_Reg = ( ( WORT2_Reg & ~0x03FF01FF ) | ( wrdly & 0x1FF ) | ( ( wordly & 0x3FF ) << 16 ) );	

		//WOR_HOLD : WOR hold RX setting when preamble/sync/carrier detected ok.
		//      [0]: No hold. [1]: Hold RX.
		//WOR_SEL: TWWS=1 setting.		
		//      [0]:RX valid packet.  [1]: preamble/sync/carrier  detected ok.
		//WOR_RST: Shall be set to [1].				
		WORT2_Reg = ( ( WORT2_Reg & ~0x03000000 ) | BY_CARRIER_DETECT_OK );
		WORT2_Reg = ( ( WORT2_Reg & ~0x04000000 ) | BIT30 ); //b30:WOR_SEL=1	
		RADIO->WORT_Item.TSEL = ENABLE; 
		DSSS1_Reg = ( ( DSSS1_Reg & ~0x03000000 ) | BIT24 | BIT25 );	//b24:WOR_HOLD=1, b25:WOR_RST=1
		RADIO->WORT2 = WORT2_Reg;
		RADIO->DSSS1 = DSSS1_Reg;

		ADC_Reg = ( ( ADC_Reg & ~0x00000010 ) | BIT4 ); //b4:ARSSI=1
		RTH_Reg = ( ( RTH_Reg & ~0x00FF0000 ) | ( ( rth & 0xFF ) << 16 ) ); //b16~b23:RTH=110 (-100dBm)
		RADIO->ADC = ADC_Reg;
		RADIO->ADC_THRESHOLD = RTH_Reg;

		RFLIB_StrobeCmd ( CMD_STBY );

		NVIC_EnableIRQ ( RADIO_IRQn );//enable RF interrupt
		INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_WAKEBB; 
		RADIO->RFINT_WRITE = INTIE_Reg;	
		

		RADIO->WORT_Item.WN = 0;
		RADIO->WORT_Item.WOR_EN = ENABLE;//enable WOR function

		//WOR_SEL=1 : preamble/sync/carrier detected ok -> TWWS go high -> wake up MCU
		//defined process by user
		//...
		//...
		//RFLIB_StrobeCmd ( CMD_STBY );  //TWWS reset by strobe command.
}

/************************************************************************
**  WOT_enable
************************************************************************/
void WOT_enable ( void )
{
		Uint16 wrdly, wordly;
	
		wrdly = 4;	
		wordly = 127;
	
		RCOSC_Cal(); //internal RC OSC CAL
		//RTC_Enable();   //use external RC OSC

		//RF_SetBBIO ( EOAC_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE ); 
		RF_SetBBIO ( TWWS_GPIO1, ARCWTR_GPIO2, ROSC_CKO, ENABLE ); 
	
		//setup WOT Sleep time 		
		//WRDLY[8:0] is 4 and above.
		//WOR Sleep  Period = (WORDLY[9:0]+1) x (32/4096), (7.8ms ~ 7.99s).
		//WORDLY[9:0]=127 : WOR Sleep  Period ~ 1s		
		WORT2_Reg = ( ( WORT2_Reg & ~0x03FF01FF ) | ( wrdly & 0x1FF ) | ( ( wordly & 0x3FF ) << 16 ) );	
		RADIO->WORT2= WORT2_Reg;		
		
		//WOR_RST: Shall be set to [1].
		DSSS1_Reg = ( ( DSSS1_Reg & ~0x02000000 ) | BIT25 ); //WOR_RST=1
		RADIO->DSSS1= DSSS1_Reg;		
		
		RFLIB_StrobeCmd ( CMD_STBY );
		WriteFIFO ( PN9_TAB, sizeof ( PN9_TAB ) ); //Write data to TX FIFO

		NVIC_EnableIRQ ( RADIO_IRQn );//enable RF interrupt
		INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_WAKEBB; 
		RADIO->RFINT_WRITE = INTIE_Reg;	

		RADIO->WORT_Item.WMODE = WOT;//select WOT function
		RADIO->WORT_Item.WN = 0;
		RADIO->WORT_Item.WOR_EN = ENABLE;//enable WOT function

		//defined process by user
		//...
		//...
		//RADIO->WORT_Item.WOR_EN = DISABLE; //disable WOT function
		//RFLIB_StrobeCmd ( CMD_STBY ); //WAKEBB reset by strobe command.
}

/************************************************************************
**  TWWS_enable
************************************************************************/
void TWWS_enable ( void )
{
		Uint16 wrdly, wordly;
	
		wrdly = 409;
		wordly = 127;
	
		RCOSC_Cal(); //internal RC OSC CAL
		//RTC_Enable();   //use external RC OSC
		
		RF_SetBBIO ( TWWS_GPIO1, TWWS_GPIO2, ROSC_CKO, ENABLE ); 

		//WOR_AC Period = (WOR_AC[8:0]+1) x (1/4096), (244us ~ 125ms).
		//WOR_SL Period = (WOR_SL[9:0]+1) x (32/4096), (7.8ms ~ 7.99s).
		//Note : TWWS function which enables this device to output a periodic square wave from GPIO to a MCU
		//                        _____     _____   _____
		//    TWWS signal : _____|   |_____|   |_____|   |
		//                    1s    1s    1s    1s    1s    1s
		//
		//    TWWS interrupt are activated by a rising edge.
		//    first time, interrupt interval=1s
		//    other times, interrupt interval=2s

	  //WRDLY[8:0]=409 : WOR Active Period ~ 0.1s, WORDLY[9:0]=127 : WOR Sleep  Period ~ 1s
		WORT2_Reg = ( ( WORT2_Reg & ~0x03FF01FF ) | ( wrdly & 0x1FF ) | ( ( wordly & 0x3FF ) << 16 ) );	
		RADIO->WORT2= WORT2_Reg;	
	
		NVIC_EnableIRQ ( RADIO_IRQn );//enable RF interrupt
		INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_WAKEBB; 
		RADIO->RFINT_WRITE = INTIE_Reg;			

		//RADIO->WORT_Item.TSEL = DUTY_BY_AC; 
		RADIO->WORT_Item.TSEL = DUTY_BY_SL; 
		RADIO->WORT_Item.TWOR = ENABLE; 

		//TWWS go high -> wake up MCU
		//defined process by user
		//...
		//...
		//RADIO->WORT_Item.TWOR = DISABLE;   //disable TWWS function
}

/************************************************************************
**  RSSI_measurement
************************************************************************/
void RSSI_measurement ( void )
{
		Uint32 volatile adc_tmp;
		//Uint32 volatile adco_tmp;
		
		RFLIB_StrobeCmd ( CMD_STBY );	  
		
		RF_SetBBIO ( FSYNC_GPIO1, ARCWTR_GPIO2, RCK_CKO, ENABLE ); 				
	
		//RXCTL_Reg = ( ( RXCTL_Reg & ~0x00010000 ) | BIT16 ); //b16:AGCE=1, AGC on
		RADIO->RXCTL = RXCTL_Reg;		
	
		//AVSEL[1:0]: [00]:AVSEL_NO_AVERAGE, [01]:AVSEL_AVERAGE_2, [10]:AVSEL_AVERAGE_4, [11]:AVSEL_AVERAGE_8,			
		ADC_Reg = ( ( ADC_Reg & ~0x00000C18 ) | BIT4 | AVSEL_AVERAGE_8 ); //b3:RADC=0, b4:ARSSI=1, AVSEL(RADC=0),ADC average is set by AVSEL		
		
		//MVSEL[1:0]: [00]:MVSEL_AVERAGE_8, [01]:MVSEL_AVERAGE_16, [10]:MVSEL_AVERAGE_32, [11]:MVSEL_AVERAGE_64,
		//ADC_Reg = ( ( ADC_Reg & ~0x00000318 ) | BIT3 | BIT4 | MVSEL_AVERAGE_8 ); //b3:RADC=1, b4:ARSSI=1, MVSEL(RADC=1),ADC average is set by MVSEL					
	
		//The ADC clock is :PF8M/1.6, ADC clock periods is 1/ADC clock
		//Each RSSI measurning time is 30 * ADC clock periods * 8(MVSEL_AVERAGE_8) + (1/PF8M)*8*5
		//IF data rate=9.6Kbps,PF8M=6.14M. Data rate=250Kbps,PF8M=8M. Else PF8M=6.4M.
		//ex:Fxtal=12.8MHz,data rate=100Kbps,PF8M=6.4M, the ADC clock is :6.4M/1.6=4MHz
		//Each RSSI measurning time is 30 * (1/4M) * 8 + (1/6.4M) * 8 * 5 = 66.25us
	
		RADIO->ADC = ADC_Reg;		
		
		NVIC_EnableIRQ ( RADIO_IRQn );		
		INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_FSYNC; 
		RADIO->RFINT_WRITE = INTIE_Reg;	
		
    RFLIB_StrobeCmd ( CMD_RX );
		
    RF_Flag = 0;

    while ( RF_Flag == 0 )   //Stay in RX mode until receiving ID code(sync ok)
    {
        adc_tmp = RADIO->ADC_THRESHOLD_READ_Item.ADC; 	
			  //adco_tmp = RADIO->ADC_THRESHOLD_READ_Item.ADCO; //read RSSI value(environment RSSI) ADCO[8:0] when AGC on		
    } 
		
	  adc_tmp = RADIO->ADC_THRESHOLD_READ_Item.ADC; 	
	  //adco_tmp = RADIO->ADC_THRESHOLD_READ_Item.ADCO; //read RSSI value(wanted signal RSSI) ADCO[8:0] when AGC on  
}

/*********************************************************************
** RF_SetBBIO
*********************************************************************/
void RF_SetBBIO ( Uint32 gio1s, Uint32 gio2s, Uint32 ckos, Uint8 sw )
{
    GPIO_Reg = ( ( GPIO_Reg & ~0x070F0F0F ) | ( ckos << 16 ) | ( gio2s << 8 ) | gio1s );		

    if ( ( Uint8 ) sw )
    {
        GPIO0->ALTFUNCSET |= BIT13 | BIT14 | BIT15; //Set GPIO0 BIT13->BB_GIO1 ,BIT14->BB_GIO2, BIT15->BB_CKO
        GPIO0->OUTENABLESET |= BIT13 | BIT14 | BIT15; //Set GPIO0 BIT13->BB_GIO1 ,BIT14->BB_GIO2, BIT15->BB_CKO
				GPIO_Reg |= BIT24 | BIT25 | BIT26; //b24:CKOIOS=1, b25:GIO2IOS=1, b26:GIO1IOS=1
    }
    else
    {
        GPIO0->ALTFUNCCLR = BIT13 | BIT14 | BIT15; //CLEAR GPIO0 BIT13->BB_GIO1 ,BIT14->BB_GIO2, BIT15->BB_CKO
        GPIO0->OUTENABLECLR = BIT13 | BIT14 | BIT15; //CLEAR GPIO0 BIT13->BB_GIO1 ,BIT14->BB_GIO2, BIT15->BB_CKO
    }
		
		RADIO->RFGIO = GPIO_Reg;		
}

/************************************************************************
**  UserRegister
************************************************************************/
void UserRegister ( void )
{
		Reg_Init();
	
		IDLength ( ID_4BYTES ); 		
	  if ( WriteID() ) 
		    Err_State(); 

		/* 
				  |	CFG1     	|	CFG2		  | CFG3		  | CFG4			| CFG5      | CFG6      | CFG7
		Frequency | 433MHz    |	433MHz		| 433MHz		| 433MHz		| 433MHz	  | 433MHz    | 433MHz
	  Data Rate	  | 9.6kbps	  |	50kbps		|	100kbps	  |	150kbps   |	250kbps   |	500kbps   |	2Mbps 
	  IFBW		  | 100KHz	  |	50KHz		  |	100KHz	  |	150KHz    |	250KHz    |	250KHz    |	250KHz 
		Fdev	  | 37.5KHz	  |	18.75KHz	|	37.5KHz	  |	56.25KHz  |	93.75KHz  |	187.5KHz  |	498.05KHz
	  Crystal     | 12.8MHz	  |	12.8MHz		|	12.8MHz	  | 19.2MHz   | 16.0MHz   | 16.0MHz   | 16.0MHz				
    
		0x50001020 , 0x0000003F, 0x0000003F, 0x0000003F, 0x0000003F, 0x0000003F, 0x0000003F, 0x0000003F  //[0020] FIFO                                
		0x50001024 , 0x00040009, 0x00030000, 0x00030000, 0x00050000, 0x000B0000, 0x000B0000, 0x000E0000  //[0024] SYCK                                
		0x50001028 , 0x07000000, 0x07000000, 0x07000000, 0x07000000, 0x07000000, 0x07000000, 0x07000000  //[0028] GPIO                                		  
		0x50001104 , 0x0007027E, 0x0007027E, 0x0007027E, 0x0007027E, 0x0007027E, 0x0007027E, 0x0007027E  //[0104] TXPWR 
		0x50001174 , 0x3D580300, 0x3D580300, 0x3D580300, 0x3D580300, 0x3D580300, 0x3D580300, 0x3D580300  //[0174] CODE1 
	
		*/
		
		//Default CFG5 250kbps

//		FIFO_Reg = 0x0000003F; //b0~b25:FEP, lengths=FEP+1  
		FIFO_Reg = 0x00000096; //b0~b25:FEP, lengths=FEP+1   151 byte
		RADIO->FIFOCTRL = FIFO_Reg;		
				
		SYCK_Reg = 0x000B0000; //b0~b7:SDR, Data rate=Data rate/(SDR+1)  
		RADIO->SYCK = SYCK_Reg;		
		
		GPIO_Reg = 0x07000000; //RFIO	 
		RADIO->RFGIO = GPIO_Reg;	
		
		TXPWR_Reg = 0x0007027E;	//b0~b5:TBG, b6~b7:PAC, b8~b9:TDC, 62/1/2 
		RADIO->TXPOWER = TXPWR_Reg;					
    
		//IDL[1:0]: [00]:2bytes, [01]:4bytes, [10]:6bytes, [11]:8bytes 
		CODE1_Reg = 0x3D580300; //b22~b23:IDL, 4bytes
		RADIO->CODE1 = CODE1_Reg;			
		
}

/************************************************************************
**  Reg_Init
************************************************************************/
void Reg_Init ( void )
{
		//CONFIG INDEX in A9139M0_RFLIB.h.		
  	CODE1_Reg = A9139M0_Config[INDEX_CODE1].load[ConfigType]; 
		FIFO_Reg = A9139M0_Config[INDEX_FIFO].load[ConfigType]; 
		TXPWR_Reg = A9139M0_Config[INDEX_TXPWR].load[ConfigType];				 	
		GPIO_Reg = A9139M0_Config[INDEX_GPIO].load[ConfigType]; 
		PWRCTL2_Reg = A9139M0_Config[INDEX_PWRCTL2].load[ConfigType]; 
		FPCTL_Reg = A9139M0_Config[INDEX_FPCTL].load[ConfigType]; 
		ADC_Reg = A9139M0_Config[INDEX_ADC].load[ConfigType];
		RTH_Reg = A9139M0_Config[INDEX_RTH].load[ConfigType];
		ADC12B_CTL_Reg = A9139M0_Config[INDEX_12BADC_CTL].load[ConfigType];
		RCCTRL1_Reg = A9139M0_Config[INDEX_RCCTL1].load[ConfigType];
		RCCTRL2_Reg = A9139M0_Config[INDEX_RCCTL2].load[ConfigType];
		INTIE_Reg = A9139M0_Config[INDEX_INTIE].load[ConfigType];
		SYN_Reg = A9139M0_Config[INDEX_SYN].load[ConfigType];
		SYCK_Reg = A9139M0_Config[INDEX_SYCK].load[ConfigType];
		RXCTL_Reg = A9139M0_Config[INDEX_RXCTL].load[ConfigType];
		DSSS1_Reg = A9139M0_Config[INDEX_DSSS1].load[ConfigType];
		WORT2_Reg = A9139M0_Config[INDEX_WORT2].load[ConfigType];
}

/************************************************************************
**  IDLength
************************************************************************/
void IDLength ( Uint32 idl )
{
		CODE1_Reg = ( ( CODE1_Reg & ~0x00C00000 ) | ( ( idl & 0x03 ) << 22 ) );			
		RADIO->CODE1 = CODE1_Reg;		
}

/************************************************************************
**  TXPower
************************************************************************/
void TXPower ( Uint32 tbg, Uint32 tdc, Uint32 pac )
{
		TXPWR_Reg = ( ( TXPWR_Reg & ~0x000003FF ) | ( tbg & 0x3F ) | ( ( tdc & 0x03 ) << 8 ) | ( ( pac & 0x03 ) << 6 ) );		
		RADIO->TXPOWER = TXPWR_Reg;		
}

/************************************************************************
**  FIFOLength
************************************************************************/
void FIFOLength ( Uint32 fep )
{
		fep--;
		FIFO_Reg = ( ( FIFO_Reg & ~0x00003FFF ) | ( fep & 0x3FFF ) );		
		RADIO->FIFOCTRL = FIFO_Reg;		
}

/************************************************************************
**  TXModeSel
************************************************************************/
void TXModeSel ( Uint8 mode )
{
    if ( mode == DIRECT )
    {
        MODEC2_Reg = MODEC2_Reg & ~0x00000002; //b1:FMS=0
    }
    else
    {
				MODEC2_Reg = ( ( MODEC2_Reg & ~0x00000002 ) | BIT1 ); //b1:FMS=1
    }
		
		RADIO->MODECTRL2 = MODEC2_Reg;		
}

/************************************************************************
**  TXModulation
************************************************************************/
void TXModulation ( Uint8 mode )
{
    if ( mode == ENABLE )
    {
         TXCTL_Reg = ( ( TXCTL_Reg & ~0x00000002 ) | BIT1 ); //TME = 1
    }
    else
    {
        TXCTL_Reg = ( TXCTL_Reg & ~0x00000002 ); //TME = 0
    }
		
		RADIO->TXCTL = TXCTL_Reg;		
}

/************************************************************************
**  CrystalSelect
************************************************************************/
void CrystalSelect ( void )
{
    switch ( ConfigType ) 
    {	
				case 3: //19.2   
				SystemCoreClock = ( 19200000UL ); 
				break;
							
				case 4://16		
				case 5:
				case 6:
				SystemCoreClock = ( 16000000UL ); 
				break;
							
				case 0://12.8
				case 1:
				case 2:
				SystemCoreClock = ( 12800000UL ); 
				break;    
    }

}

/************************************************************************
**  WDTReset
************************************************************************/
void WDTReset ( void )
{
		/* Watchdog reset */		
    WATCHDOG->LOCK = 0x1ACCE551;    // unlock
    WATCHDOG->LOAD = SystemCoreClock / 100; 
		WATCHDOG->CTRL = 0x03; //b0:INTIE=1, b1:RESTEN = 1	
    WATCHDOG->LOCK = 0;             // lock	
	
}

/************************************************************************
**  AES_CCM_mode
************************************************************************/
void AES_CCM_mode ( void )
{
    Uint8  i; 
		Uint16 AES_CCM_FIFO_Length;    

	  for ( i = 0; i < 4; i++ )
				AESCCM->AESKEY[i] = AES_KEY_IN[i];

	  for ( i = 0; i < 5; i++ )
				AESCCM->NONCE[i] = NONCE[0][i];	//CCM Nonce Setting level 4~7 => 0~3

    AES_CCM_FIFO_Length = 23;  
  
		RADIO->TXPKT  = 0x00000003;	//TX_PACKET Setting=0~3
	    
		NVIC_EnableIRQ ( RADIO_IRQn );		
		INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_WTR;
		RADIO->RFINT_WRITE = INTIE_Reg;	
	
    FIFOLength ( AES_CCM_FIFO_Length );

    if( Flag_Master )
    {   
				RF_SetBBIO ( FMTDO_GPIO1, TMEO_GPIO2, DCK_CKO, ENABLE ); 

        while( 1 )
        {
						AESCCM->CONTROLB [0] = BIT0 | BIT5;	//b0:CCM_TX_ON=1, b5:CCM_RX_ON=1

						WriteFIFO( CCM_FIFO, sizeof ( CCM_FIFO ) );	
						
						NVIC_EnableIRQ ( AES_CCM_IRQn );
						AESCCM->CONTROLB [3] = BIT6 | BIT7;	//b6:AESIE=1, b7:CCMIE=1
						AESCCM->CONTROLB [0] = BIT0 | BIT5 | BIT7; //b0:CCM_TX_ON=1, b5:CCM_RX_ON=1, b7:TXEN_GO=1
						RF_CCM_Flag = 0;
						while ( RF_CCM_Flag == 0 )
						    ;
            RFLIB_StrobeCmd ( CMD_TX );
            RF_Flag = 0;

            while ( RF_Flag == 0 );    //wait transmit completed

            Delay10ms ( 1 );

        }
    }
    else
    {
				RF_SetBBIO( TRXD_GPIO1, FSYNC_GPIO2, RCK_CKO, ENABLE ); 

        while( 1 )
        {
						AESCCM->CONTROLB [0] = BIT0 | BIT5;	//b0:CCM_TX_ON=1, b5:CCM_RX_ON=1
					
					  RFLIB_StrobeCmd( CMD_RX );					
            RF_Flag = 0;
            while( RF_Flag == 0 );    //wait receive completed
						
						NVIC_EnableIRQ ( AES_CCM_IRQn );
						AESCCM->CONTROLB [3] = BIT6 | BIT7;	//b6:AESIE=1, b7:CCMIE=1
						AESCCM->CONTROLB [0] = BIT0 | BIT5 | BIT6; //b0:CCM_TX_ON=1, b5:CCM_RX_ON=1, b7:RXDE_GO=1
						RF_CCM_Flag = 0;	
						while ( RF_CCM_Flag == 0 )
						    ;					
            if ( AESCCM->CONTROLB[2] & CCM_FAIL ) 
            {
                Delay100us ( 1 );
                while ( 1 );
            }
						
            if ( AESCCM->CONTROLB[2] & CCM_OK ) 
            {
                
            }
						
            Delay100us ( 1 );

        }

    }

}

/************************************************************************
**  ACK_mode
************************************************************************/
void ACK_mode ( void )
{		
		NVIC_EnableIRQ ( RADIO_IRQn );		
		INTIE_Reg	= RADIO_INT_CLEAR | ENABLE_RFINT_VPOAK;
		RADIO->RFINT_WRITE = INTIE_Reg;	
	
		PWRCTL2_Reg = ( ( PWRCTL2_Reg & ~0x00800000 ) | BIT23 );
    POWER->PWR_TA = 0xAAAAAAAA;
    POWER->PWR_TA = 0x55555555;
		POWER->POWERCTRL2 = PWRCTL2_Reg; //b23:SPSS=1

    if ( Flag_Master )
    {	
				RADIO->RXDEM3_Item.EARKS = 1;	//auto resnt, auto ack
				RADIO->RXDEM3_Item.ARC = 15;	//resent number
				RADIO->RXDEM3_Item.ARTMS = 1;	//0 random, 1 fix interval
				RADIO->RXDEM3_Item.VKP = 0;	//0 10us, 1 30us, by pulse
				RADIO->RXDEM3_Item.VKM = 0;	//0 by event, 1 by pulse
				RADIO->RXDEM3_Item.ARD = 30;
				RADIO->RXDEM3_Item.RND = 9;	//random seed			
	
				RF_SetBBIO ( ARCWTR_GPIO1, VPOAK_GPIO2, DCK_CKO, ENABLE ); 

        while ( 1 )
        {            
						WriteFIFO ( PN9_TAB, sizeof ( PN9_TAB ) ); //Write data to TX FIFO
					
            RFLIB_StrobeCmd ( CMD_TX );

            while ( GIO1 == 0 );     //wait TX ready
            while ( GIO1 == 1 );     //wait transmit completed

            Delay100us ( 1 );
            RFLIB_StrobeCmd ( CMD_STBY );

            Delay10ms ( 1 );

        }

    }
    else
    {        
				RADIO->RXDEM3_Item.EARKS = 1;	//auto resnt, auto ack
			
				RF_SetBBIO ( TRXD_GPIO1, ARCWTR_GPIO2, RCK_CKO, ENABLE );         
				
        while ( 1 )
        {
            RFLIB_StrobeCmd ( CMD_RX );
						
            while ( GIO2 == 0 );     
            while ( GIO2 == 1 );     //wait receive completed.						
					
            Delay100us ( 1 );  //delay 200us for TX ramp down

            RFLIB_StrobeCmd ( CMD_STBY );

        }

    }
}

/************************************************************************
**  ACK_FIFO_mode
************************************************************************/
void ACK_FIFO_mode ( void )
{
		Uint8 ackfep;
		
		ackfep = 63;
	
		NVIC_EnableIRQ ( RADIO_IRQn );		
		INTIE_Reg	|= RADIO_INT_CLEAR | ENABLE_RFINT_VPOAK;
		RADIO->RFINT_WRITE = INTIE_Reg;
	
		PWRCTL2_Reg = ( ( PWRCTL2_Reg & ~0x00800000 ) | BIT23 );
    POWER->PWR_TA = 0xAAAAAAAA;
    POWER->PWR_TA = 0x55555555;
		POWER->POWERCTRL2 = PWRCTL2_Reg; //b23:SPSS=1
	
		SYN_Reg = ( ( SYN_Reg & ~0x00FF0001 ) | BIT0 | ( ackfep << 16 ) );    
		RADIO->SYN = SYN_Reg; //b0:EAF=1, b16~b23:ACKFEP=63
	
    WriteFIFO ( PN9_TAB, sizeof( PN9_TAB ) ); //Write data to TX FIFO

    if ( Flag_Master )
    {		
				RADIO->RXDEM3_Item.EARKS = 1;	//auto resnt, auto ack
				RADIO->RXDEM3_Item.ARC = 15;	//resent number
				RADIO->RXDEM3_Item.ARTMS = 1;	//0 random, 1 fix interval
				RADIO->RXDEM3_Item.VKP = 0;	//0 10us, 1 30us, by pulse
				RADIO->RXDEM3_Item.VKM = 0;	//0 by event, 1 by pulse
				RADIO->RXDEM3_Item.ARD = 30;
				RADIO->RXDEM3_Item.RND = 9;	//random seed				
			
				RF_SetBBIO ( ARCWTR_GPIO1, VPOAK_GPIO2, DCK_CKO, ENABLE ); 

        while(1)
        {				
						RF_SetBBIO ( FMTDO_GPIO1, TMEO_GPIO2, VPOAK_CKO, ENABLE ); 

            RFLIB_StrobeCmd ( CMD_TX );

            while ( GIO2 == 0 );     //wait TX ready
            while ( GIO2 == 1 );     //wait transmit completed

            RF_SetBBIO ( FSYNC_GPIO1, TRXD_GPIO2, RCK_CKO, ENABLE ); 

            while ( GIO1 == 0 );     //wait sync ok
            while ( GIO1 == 1 );     //wait receive completed

						RF_SetBBIO ( FSYNC_GPIO1, VPOAK_GPIO2, RCK_CKO, ENABLE ); 

            Delay100us ( 1 );
            RFLIB_StrobeCmd ( CMD_STBY );
            Delay10ms ( 1 );

        }

    }
    else
    {      	
				RADIO->RXDEM3_Item.EARKS = 1;	//auto resnt, auto ack
				RADIO->RXDEM3_Item.ARC = 15;	//resent number
				RADIO->RXDEM3_Item.ARTMS = 1;	//0 random, 1 fix interval
				RADIO->RXDEM3_Item.VKP = 0;	//0 10us, 1 30us, by pulse
				RADIO->RXDEM3_Item.VKM = 0;	//0 by event, 1 by pulse
			
				RF_SetBBIO ( TRXD_GPIO1, FSYNC_GPIO2, RCK_CKO, ENABLE ); 

        while ( 1 )
        {
						RF_SetBBIO ( TRXD_GPIO1, FSYNC_GPIO2, RCK_CKO, ENABLE ); 
					
            RFLIB_StrobeCmd ( CMD_RX );

            while ( GIO2 == 0 );     //wait sync ok
            while ( GIO2 == 1 );     //wait receive completed.

						RF_SetBBIO ( TRXD_GPIO1, TMEO_GPIO2, RCK_CKO, ENABLE ); 

            while ( GIO2 == 0 );     //wait TX ready

						RF_SetBBIO ( FMTDO_GPIO1, TMEO_GPIO2, RCK_CKO, ENABLE ); 

            while ( GIO2 == 1 );     //wait transmit completed.

            RF_SetBBIO ( FMTDO_GPIO1, VPOAK_GPIO2, RCK_CKO, ENABLE ); 

            Delay100us ( 1 );  //delay 200us for TX ramp down

            RFLIB_StrobeCmd ( CMD_STBY );

        }
    }

}

/************************************************************************
**  FCC
************************************************************************/
void FCC( void )
{
    RF_SetBBIO ( ARCWTR_GPIO1, EOAC_GPIO2, DCK_CKO, ENABLE ); 
	  NVIC_EnableIRQ ( RADIO_IRQn );
		RADIO->RFINT_WRITE |= RADIO_INT_CLEAR | ENABLE_RFINT_WTR; //WTR INT	
		FIFOLength ( 64 );	//64Byte
		WriteFIFO ( PN9_TAB, sizeof ( PN9_TAB ) ); //Write data to TX FIFO
	
		RFLIB_SetCH ( 0 );

		RFLIB_StrobeCmd ( CMD_PLL );

		while(1)
		{  	
				RF_Flag = 0;
				RFLIB_StrobeCmd ( CMD_TX ); //entry tx & transmit
				while ( RF_Flag == 0 ); //wait TX finish 		  

				Delay10us ( 1 );
		}

		//Direct Tx or Rx
		//TXModeSel ( DIRECT );	//direct mode
		//TXModulation ( DISABLE );
		//RFLIB_StrobeCmd ( CMD_TX );
		//RFLIB_StrobeCmd ( CMD_RX );
		//while( 1 );
}

/************************************************************************
** TIMER0_Handler				
************************************************************************/
void TIMER0_Handler( void )
{
    timer++;
		if( timer == TIMEOUT )
        TimeoutFlag = 1;
	
		TimerCnt++;
//    if( TimerCnt >= 500 )
//    {
//        TimerCnt = 0;
//			  CmdBuf[0] = 0xF1;
//			
//				CmdBuf[1] = ( RxCnt & 0xFF00 ) >> 8;
//				CmdBuf[2] = RxCnt;
//							
//				CmdBuf[3] = ( LossCnt & 0xFF00 ) >> 8;
//				CmdBuf[4] = LossCnt;		

//				CmdBuf[5] = ( ErrCnt & 0xFF00 ) >> 8;
//				CmdBuf[6] = ErrCnt;				
//				
//				CmdBuf[7] = ( Err_BitCnt & 0xFF000000 ) >> 24;
//				CmdBuf[8] = ( Err_BitCnt & 0x00FF0000 ) >> 16;
//				CmdBuf[9] = ( Err_BitCnt & 0x0000FF00 ) >> 8;
//				CmdBuf[10] = Err_BitCnt;

//        UartSendCnt = 12;
//		    Uartptr = &CmdBuf[0];
//        UART0->DATA = CmdBuf[0];
//    }

    TIMER0->INTCLEAR = 1;
	  return;
}

/************************************************************************
** UART0_Handler	
************************************************************************/
void UART0_Handler( void ) //UARTTRX0:NMI=1
{
    if( UART0->INTSTATUS & 0x01 ) //TX
	  {
		    UART0->INTCLEAR = 0x01; //TI clear	
		    UartSendCnt--;
		
		    if ( UartSendCnt != 0 )
		    {
		        Uartptr++;
			      UART0->DATA = *Uartptr;
		    }
	
		    return;
	  }
	
    if ( UART0->INTSTATUS & 0x02 ) //RX
	  {
		    UART0->INTCLEAR = 0x02; //RI clear
			TXbuf[UartRecevicCnt]=UART0->DATA;
			UartRecevicCnt++;
		    return;
	  }
}

/*********************************************************************
** RADIO_Handler
*********************************************************************/
void RADIO_Handler ( void )
{
		if( ( RADIO->RFINT_READ_Item.WTR_INT_STATE ) == ON )	
		{
				RF_Flag = 1;				
		}	  
		
		if( ( RADIO->RFINT_READ_Item.WAKEBBINT_STATE ) == ON )	
		{
				//RF_Flag = 1;
		}
		
		if( ( RADIO->RFINT_READ_Item.FPFIE_STATE ) == ON )	
		{
				//RF_Flag = 1;
		}
		
		if( ( RADIO->RFINT_READ_Item.FSYNCIE_STATE ) == ON )	
		{
				RF_Flag = 1;
		}
		
		if( ( RADIO->RFINT_READ_Item.VPOAKIE_STATE ) == ON )	
		{
				RF_Flag = 1;
		}
				
		INTIE_Reg	|= RADIO_INT_CLEAR ;		
		RADIO->RFINT_WRITE = INTIE_Reg;	
}

/*********************************************************************
** NMI_Handler
*********************************************************************/
void NMI_Handler ( void )
{
		
}

/*********************************************************************
** AES_CCM_Handler
*********************************************************************/
void AES_CCM_Handler( void )
{    
	  if( AESCCM->CONTROLB[1] & CCMI_STATE ) 
		    RF_CCM_Flag = 1;
		  
	  if( AESCCM->CONTROLB[1] & AESI_STATE ) 
		    RF_AES_Flag = 1;
		
	  AESCCM->CONTROLB[1] |= AESCCM_Int_Clear; 
}
    
/*********************************************************************
** EnableCache
*********************************************************************/
void EnableCache ( void )
{
		// Enable the cache in auto power and auto invalidate mode (CCR[0].EN = 1), enable the statistics logic CCR[6].STATISTIC_EN = 1, enable prefetch (CCR[5].SET_PREFETCH = 1)	
		CACHE->CCRItem.EN = ENABLE;
		CACHE->CCRItem.STATISTIC_EN = ENABLE;
		CACHE->CCRItem.SET_PREFETCH = ENABLE;		
		// Poll till the cache is enabled (SR[1:0].CS=1)
		while ( CACHE->SRItem.CS == CACHE_ENABLING );		
}

/*********************************************************************
** DisableCache
*********************************************************************/
void DisableCache ( void )
{
		// Disable the cache in auto power and auto invalidate mode (CCR[0].EN = 0), disable the statistics logic CCR[6].STATISTIC_EN = 0, disable prefetch (CCR[5].SET_PREFETCH = 0)
		CACHE->CCRItem.EN = DISABLE;
		CACHE->CCRItem.STATISTIC_EN = DISABLE;
		CACHE->CCRItem.SET_PREFETCH = DISABLE;		
		// Poll till the cache is disabled (SR[1:0].CS=0)		
		while ( CACHE->SRItem.CS == CACHE_DIABLING );
}

/*********************************************************************
** EnableHS
*********************************************************************/
void EnableHS ( void )
{			
		RADIO->CLOCKCTRL = 0x0011290F;
		POWER->XRC_CTRL = 0x0200C0C1;			
	
		if( (FLASH->WAIT_STATE_Item.IS_PLL_Mode) == 0 )
		{
				//ws_cfg(for "new ws_cfg")
				//EN_PLL + ws_cfg
				//IS_PLL_Mode + EN_PLL + ws_cfg
				FLASH->WAIT_STATE = 0x00000003;
				FLASH->WAIT_STATE = 0x00000007;
				FLASH->WAIT_STATE = 0x0000002F;//b0~1:ws_cfg=3, b2:EN_PLL=1, b3:IS_PLL_Mode=1, b5:MCUSWS=1	
		}
}

/*********************************************************************
** DisableHS
*********************************************************************/
void DisableHS ( void )
{
		//IS_PLL_Mode + EN_PLL + ws_cfg	
		//EN_PLL + ws_cfg
		//ws_cfg(for "new ws_cfg")				
		FLASH->WAIT_STATE = 0x00000007;
		FLASH->WAIT_STATE = 0x00000003;
		FLASH->WAIT_STATE = 0x00000000;//b0~1:ws_cfg=0, b2:EN_PLL=0, b3:IS_PLL_Mode=0	
}
